(**************************************************************************)
(*                                                                        *)
(*  This file is part of Frama-C.                                         *)
(*                                                                        *)
(*  Copyright (C) 2013-2014                                               *)
(*    CEA (Commissariat à l'énergie atomique et aux énergies              *)
(*         alternatives)                                                  *)
(*                                                                        *)
(*  You may redistribute it and/or modify it under the terms of the GNU   *)
(*  Lesser General Public License as published by the Free Software       *)
(*  Foundation, version 2.1.                                              *)
(*                                                                        *)
(*  It is distributed in the hope that it will be useful, but WITHOUT     *)
(*  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY    *)
(*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General      *)
(*  Public License for more details.                                      *)
(*                                                                        *)
(*  See the GNU Lesser General Public License version 2.1 for more        *)
(*  details (enclosed in the file LICENSE).                               *)
(*                                                                        *)
(**************************************************************************)

let putenv_default name value =
  try
    ignore (Unix.getenv name)
  with Not_found ->
    Unix.putenv name value

let setup_env () =
  putenv_default "CC"        "gcc";
  putenv_default "CCFLAGS"   "";
  putenv_default "CPP"       "gcc -E -P";
  putenv_default "CPPFLAGS"  ""

let usage_short ="Usage: covlabels [-update|-check|-init|-stats] file.c [-drivers GLOB] [-main FUN]

Compute lavel coverage based on a set of test drivers.
"
let help_details ="
The key option is '-drivers'. It specifies a pattern to find test drivers.
The pattern may contain *, ?, [...] like the shell, and variables of the form
${NAME} where NAME is one of SOURCE, DIRNAME, BASENAME, BASENAME_NO_EXT,
and MAINFUN. The default value is specific to PathCrawler:
  ${DIRNAME}/testcases_${BASENAME_NO_EXT}/${MAINFUN}/testdrivers/TC_*.c.
  
The MAINFUN variable indicates the function under test (-main in Frama-C). By
default it's set to *, that is every possible function for which PathCrawler
had been run. You may change it via the '-main' flag.

You may override the default C compiler (also used as linker, default: gcc)
and C preprocessor (default: gcc -E -P) with environment variables CC and CPP. You
may also specify additional flags with CCFLAGS and CPPFLAGS.
"

type mode = Update | Check | Init | Stats

let on_source mainfun driverpatt mode force source =
  let labelsfile = (Filename.chop_extension source)^".labels" in
  match mode with
  | Init ->
    LabelActions.init labelsfile source force
  | Check ->
    let report = Coverage.coverage ~force mainfun driverpatt source in
    ignore (LabelActions.check labelsfile report)
  | Update ->
    let report = Coverage.coverage ~force mainfun driverpatt source in
    let labelstable = LabelActions.update labelsfile report in
    LabelActions.stats labelstable
  | Stats ->
    if Sys.file_exists labelsfile then
      let data = Data.create () in
      Data.load data labelsfile;
      LabelActions.stats data
    else
      Format.eprintf "[covlabels] %s file does not exist@." labelsfile

let main ?current argv = 
  setup_env ();
  let usage = usage_short^"\nOptions:" in
  let mainfun = ref "*"
  and driverpatt = ref "${DIRNAME}/testcases_${BASENAME_NO_EXT}/${MAINFUN}/testdrivers/*.c"
  and rev_sources = ref []
  and mode = ref Update
  and force = ref false in
  let spec = Arg.align [
    "-update", Arg.Unit (fun () -> mode := Update), " Compute coverage and update .labels (action by default)";
    "-check", Arg.Unit (fun () -> mode := Check), " Compute coverage and check .labels";
    "-init", Arg.Unit (fun () -> mode := Init)," Initialize .labels (do not run any test)";
    "-stats", Arg.Unit (fun () -> mode := Stats)," Print stats about .labels (do not run any test)";
    "-drivers", Arg.Set_string driverpatt, "GLOB Set driver file pattern (PathCrawler-specific default)";
    "-main", Arg.Set_string mainfun, "FUN Set main function name (default: *)";
    "-force", Arg.Set force, " Force recomputations (for -check and -update) or overwriting (for -init)"
  ]
  and add_sources source =
    rev_sources := source :: !rev_sources
  in

  try
    Arg.parse_argv ?current argv spec add_sources usage;
    if !rev_sources = [] then
      (print_endline usage_short;
      print_endline "Run `covlabels --help` for more information.")
    else
      List.iter (on_source !mainfun !driverpatt !mode !force) (List.rev !rev_sources)
  with Arg.Bad error ->
    prerr_endline error
  | Arg.Help msg ->
    prerr_string msg;
    prerr_endline help_details
    
let () = main Sys.argv
