(**************************************************************************)
(*                                                                        *)
(*  This file is part of Frama-C.                                         *)
(*                                                                        *)
(*  Copyright (C) 2013-2014                                               *)
(*    CEA (Commissariat à l'énergie atomique et aux énergies              *)
(*         alternatives)                                                  *)
(*                                                                        *)
(*  You may redistribute it and/or modify it under the terms of the GNU   *)
(*  Lesser General Public License as published by the Free Software       *)
(*  Foundation, version 2.1.                                              *)
(*                                                                        *)
(*  It is distributed in the hope that it will be useful, but WITHOUT     *)
(*  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY    *)
(*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General      *)
(*  Public License for more details.                                      *)
(*                                                                        *)
(*  See the GNU Lesser General Public License version 2.1 for more        *)
(*  details (enclosed in the file LICENSE).                               *)
(*                                                                        *)
(**************************************************************************)

type row = {
  id: int;
  covered: bool;
  tag: string;
  loc: string;
  driver: string;
}

(**
  Detailed report
*)
type detailed_report = row list

(**
  Simplified report
*)
type report = (int, row) Hashtbl.t


let escape_slash =
  let re = Str.regexp "[/\\.]" in
  Str.global_replace re "_"

let newer_than file1 file2 =
  try
    let stat1 = Unix.stat file1 and stat2 = Unix.stat file2 in
    stat1.Unix.st_mtime > stat2.Unix.st_mtime
  with Unix.Unix_error (Unix.ENOENT, _, _) ->
    false

let compile_test_driver file binary output_file =
  let compile_cmd = String.concat " " [
    "$CC";
    "$CPPFLAGS $CCFLAGS";
    "-include "^StrUtils.shell_escape (ForeignCode.runtime_h ());
    "-DCOVLABELS_OUTPUT="^StrUtils.shell_escape output_file;
    "-DCOVLABELS_TESTDRIVER="^StrUtils.shell_escape file;
    StrUtils.shell_escape file;
    "-o "^StrUtils.shell_escape binary;
    "2>&1 > "^StrUtils.shell_escape (binary^".log")
  ] in
  Format.eprintf "[covlabels] compile test driver %s@." binary;
(*  prerr_endline compile_cmd;*)
  let status = Sys.command compile_cmd in
  if status <> 0 then
    failwith "Cannot compile test driver"

let run_test_driver binary =
  Format.eprintf "[covlabels] run test driver %s@." binary;
  let status = Sys.command (StrUtils.shell_escape binary) in
  if status <> 0 then
    failwith "Cannot run test driver successufully"

let load_test_driver_coverage file driver =
  let data = ref [] in
  let f linenum fields =
    try
      match fields with
      | idstr :: statusstr :: tag :: loc :: [] -> 
        let id = int_of_string idstr
        and covered = 0 != int_of_string statusstr in
        data := {id; covered; tag; loc; driver} :: !data
      | _ ->
        Format.eprintf "[covlabels warning] invalid row (3 fields expected) at %s:%d@." file linenum
    with Invalid_argument _ ->
      Format.eprintf "[covlabels warning] invalid row at %s:%d@." file linenum
  in
  Csv.read_file f file;
  !data

let individual_coverage ?(force=false) ~driver:file ~outdir:tcdir =
  let binary = Filename.concat tcdir (escape_slash (Filename.chop_extension file)) in
  let output_file = binary^".covlabels" in
  if not force && newer_than binary file then
    Format.eprintf "[covlabels] skip test driver compilation %s@." binary
  else
    compile_test_driver file binary output_file;  
  if not force && newer_than output_file binary then
    Format.eprintf "[covlabels] skip test driver execution %s@." binary
  else
    run_test_driver binary;
  load_test_driver_coverage output_file file

(** Put into a hash table, to remove id duplicate keep covered rather than reached *)
let simplify data =
  let h = Hashtbl.create 100 in
  let add row =
    let id = row.id in
    if not (Hashtbl.mem h id) then
      Hashtbl.add h id row
    else if row.covered && let row' = Hashtbl.find h id in not (row'.covered) then
      Hashtbl.replace h id row
  in
  List.iter add data;
  h

(** Print two lines *)
let preliminary_stats data =
  let reached = ref 0 in
  let covered = ref 0 in
  Hashtbl.iter (fun _id row -> incr reached; if row.covered then incr covered) data;
  Format.eprintf "[covlabels] reached labels: %d\n[covlabels] covered labels: %d@."
    !reached !covered


let coverage ?(force=false) mainfun driverpatt source =
  let dirname = Filename.dirname source in
  let dirname = if dirname = "" then "." else dirname in
  let basename = Filename.basename source in
  let basename_no_ext = Filename.chop_extension basename in
  let vartbl = Hashtbl.create 10 in
  Hashtbl.add vartbl "SOURCE" source;
  Hashtbl.add vartbl "DIRNAME" dirname;
  Hashtbl.add vartbl "BASENAME" basename;
  Hashtbl.add vartbl "BASENAME_NO_EXT" basename_no_ext;
  Hashtbl.add vartbl "MAINFUN" mainfun;
  let driverpatt = StrUtils.apply_template_hashtbl vartbl driverpatt in
  Format.eprintf "[covlabels] compute coverage for %s with tests\n  %s@." source driverpatt;
  let tcdir = Filename.concat dirname ("covlabels_"^basename_no_ext) in
  if not (Sys.file_exists tcdir) then Unix.mkdir tcdir 0o777;
  let individuals = ref [] in
  StrUtils.glob ~sort:true (fun driver ->
    individuals := individual_coverage ~force ~driver ~outdir:tcdir :: !individuals
  ) driverpatt;
  let everything = List.concat !individuals in
  let table = simplify everything in
  preliminary_stats table;
  table

